<?php
/* --------------------------------------------------------------
  ConfigFactory.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit\Factories;

use \StyleEdit\Entities\StoredBoilerplateConfig;
use \StyleEdit\Entities\StoredConfig;
use \StyleEdit\Entities\Config;
use \StyleEdit\JsonTransformer;

/**
 * Class ConfigFactory
 *
 * @package StyleEdit\Factories
 */
class ConfigFactory
{
	/**
	 * Creates a Config object.
	 *
	 * @param array $jsonData
	 *
	 * @return \StyleEdit\Entities\Config
	 */
	public function createConfig(array $jsonData)
	{
		return new Config($jsonData);
	}
	
	
	/**
	 * Creates a StoredConfig object.
	 *
	 * @param string $p_filepath
	 *
	 * @throws \InvalidArgumentException if the given filepath is not a string or the referenced file does not exist.
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function createStoredConfig($p_filepath)
	{
		$this->_checkFilepathArgument($p_filepath);
		
		$storedConfig = new StoredConfig(JsonTransformer::decode(file_get_contents($p_filepath)),
		                                 basename($p_filepath));
		
		return $storedConfig;
	}
	
	
	/**
	 * Creates a StoredBoilerplateConfig object.
	 *
	 * @param string $p_filepath
	 *
	 * @throws \InvalidArgumentException if the given filepath is not a string or the referenced file does not exist.
	 *
	 * @return \StyleEdit\Entities\StoredBoilerplateConfig
	 */
	public function createStoredBoilerplateConfig($p_filepath)
	{
		$this->_checkFilepathArgument($p_filepath);
		
		$storedBoilerplateConfig = new StoredBoilerplateConfig(JsonTransformer::decode(file_get_contents($p_filepath)),
		                                                       basename($p_filepath));
		
		return $storedBoilerplateConfig;
	}
	
	
	/**
	 * Checks if the given filepath argument is a non-empty string and if the referenced file exists.
	 *
	 * @throws \InvalidArgumentException if the given filepath is not a string or the referenced file does not exist.
	 *
	 * @param string $p_filepath
	 */
	private function _checkFilepathArgument($p_filepath)
	{
		if(!is_string($p_filepath) || empty($p_filepath))
		{
			throw new \InvalidArgumentException('Invalid $p_filepath argument value (string expected): '
			                                    . print_r($p_filepath, true));
		}
		
		if(!file_exists($p_filepath))
		{
			throw new \InvalidArgumentException('The file "' . $p_filepath . '" does not exist."');
		}
	}
}